<?php

namespace Database\Seeders;

use Illuminate\Database\Seeder;
use App\Models\User;
use App\Models\Service;
use App\Models\Consultant;
use App\Models\Booking;
use App\Models\Testimonial;
use App\Models\AstrologyProfile;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Artisan;
use Spatie\Permission\Models\Role;

class DatabaseSeeder extends Seeder
{
    public function run(): void
    {
        // 1. SETUP ROLES & PERMISSIONS (Filament Shield)
        // We run the shield command programmatically to ensure roles exist
        Artisan::call('shield:generate --all');

        echo "Roles generated...\n";

        // 2. CREATE SUPER ADMIN
        $admin = User::firstOrCreate(
            ['email' => 'admin@vedic.com'],
            [
                'name' => 'Super Admin',
                'password' => Hash::make('password'),
                'email_verified_at' => now(),
            ]
        );
        $admin->assignRole('super_admin');

        // 3. CREATE CONSULTANTS (Astrologers)
        $consultant1 = User::create([
            'name' => 'Acharya Shastri',
            'email' => 'shastri@vedic.com',
            'password' => Hash::make('password'),
            'email_verified_at' => now(),
        ]);
        $consultant1->assignRole('panel_user');

        Consultant::create([
            'user_id' => $consultant1->id,
            'specialization' => 'Vedic Astrology & Vastu',
            'experience_years' => 15,
            'bio' => 'Expert in Kundali reading and Vastu Shastra consultation with over 15 years of experience in Bhaktapur.',
            'hourly_rate' => 1500,
            'is_active' => true,
        ]);

        $consultant2 = User::create([
            'name' => 'Pandit Sharma',
            'email' => 'sharma@vedic.com',
            'password' => Hash::make('password'),
            'email_verified_at' => now(),
        ]);
        $consultant2->assignRole('panel_user');

        Consultant::create([
            'user_id' => $consultant2->id,
            'specialization' => 'Numerology & Gemology',
            'experience_years' => 8,
            'bio' => 'Certified Gemologist and Numerology expert helping people find their lucky stones and numbers.',
            'hourly_rate' => 1200,
            'is_active' => true,
        ]);

        // 4. CREATE SERVICES
        $services = [
            [
                'name' => 'Complete Horoscope Analysis',
                'slug' => 'horoscope-analysis',
                'description' => 'Detailed analysis of your birth chart covering career, health, marriage, and finance.',
                'price' => 1500,
                'duration_minutes' => 45,
                'category' => 'Astrology',
                'image' => null, // You can put a path here later if you have one
                'is_active' => true,
            ],
            [
                'name' => 'Vastu Consultation (Home)',
                'slug' => 'vastu-home',
                'description' => 'On-site or video consultation for your home Vastu to bring peace and prosperity.',
                'price' => 5000,
                'duration_minutes' => 60,
                'category' => 'Vastu',
                'image' => null,
                'is_active' => true,
            ],
            [
                'name' => 'Marriage Matching (Kundali Milan)',
                'slug' => 'marriage-matching',
                'description' => 'Compatibility check for couples based on Ashta Koota Milan.',
                'price' => 1000,
                'duration_minutes' => 30,
                'category' => 'Matchmaking',
                'image' => null,
                'is_active' => true,
            ],
            [
                'name' => 'Business Name Selection',
                'slug' => 'business-name',
                'description' => 'Lucky name selection for your new business venture based on numerology.',
                'price' => 2500,
                'duration_minutes' => 30,
                'category' => 'Numerology',
                'image' => null,
                'is_active' => true,
            ],
        ];

        foreach ($services as $svc) {
            Service::updateOrCreate(['slug' => $svc['slug']], $svc);
        }

        // 5. CREATE CLIENTS (Regular Users)
        $client = User::create([
            'name' => 'Rajesh Karki',
            'email' => 'rajesh@gmail.com',
            'password' => Hash::make('password'),
            'email_verified_at' => now(),
        ]);
        // Note: Clients don't need a specific role in your system, or you can create a 'client' role if needed.

        // Create Profile for Client (So Dashboard Works)
        AstrologyProfile::create([
            'user_id' => $client->id,
            'full_name' => 'Rajesh Karki',
            'dob' => '1995-08-15',
            'tob' => '14:30:00',
            'pob' => 'Kathmandu, Nepal',
            'gender' => 'male',
        ]);

        // 6. CREATE BOOKINGS (History & Upcoming)
        $serviceRef = Service::first();

        // Past Completed Booking
        Booking::create([
            'user_id' => $client->id,
            'consultant_id' => $consultant1->consultant->id,
            'service_id' => $serviceRef->id,
            'scheduled_at' => now()->subDays(5),
            'status' => 'completed',
            'payment_status' => 'paid',
            'payment_method' => 'esewa',
            'meeting_link' => 'https://meet.google.com/abc-defg-hij',
        ]);

        // Future Pending Booking
        Booking::create([
            'user_id' => $client->id,
            'consultant_id' => $consultant2->consultant->id, // Different consultant
            'service_id' => $serviceRef->id,
            'scheduled_at' => now()->addDays(2),
            'status' => 'pending',
            'payment_status' => 'unpaid',
            'payment_method' => 'manual',
        ]);

        // 7. CREATE TESTIMONIALS
        Testimonial::create([
            'user_id' => $client->id,
            'message' => 'The consultation was eye-opening. Acharya ji predicted my job change accurately!',
            'rating' => 5,
            'is_active' => true,
            'is_featured' => true,
        ]);

        Testimonial::create([
            'user_id' => $client->id,
            'message' => 'Great service, but the video call had some network issues. Advice was solid though.',
            'rating' => 4,
            'is_active' => true,
            'is_featured' => true,
        ]);

        echo "Database seeding completed successfully!\n";
        echo "Super Admin: admin@vedic.com / password\n";
        echo "Consultant: shastri@vedic.com / password\n";
        echo "Client: rajesh@gmail.com / password\n";
    }
}
