<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Validation\Rules\Password;
use App\Models\AstrologyProfile;

class ProfileController extends Controller
{
    // Show the Edit Form
    public function edit()
    {
        return view('profile.edit', [
            'user' => Auth::user()
        ]);
    }

    // Update Profile Info
    public function update(Request $request)
    {
        $user = Auth::user();

        // 1. Validation (Combined User + Astrology)
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|email|unique:users,email,' . $user->id,

            // Astrology Fields
            'dob' => 'nullable|date',
            'tob' => 'nullable',
            'pob' => 'nullable|string|max:255',
            'gender' => 'nullable|string|in:male,female,other',

            // Password Fields
            'current_password' => 'nullable|required_with:new_password',
            'new_password' => ['nullable', 'confirmed', Password::defaults()],
        ]);

        // 2. Update User Basic Info
        $user->name = $validated['name'];
        $user->email = $validated['email'];

        // 3. Update Password (if provided)
        if ($request->filled('current_password')) {
            if (!Hash::check($request->current_password, $user->password)) {
                return back()->withErrors(['current_password' => 'Current password does not match.']);
            }
            $user->password = Hash::make($request->new_password);
        }

        $user->save();

        // 4. Update Astrology Profile
        if ($request->filled('dob')) {
            AstrologyProfile::updateOrCreate(
                ['user_id' => $user->id], // Find existing record by User ID
                [
                    'full_name' => $user->name, // Sync name
                    'dob' => $request->dob,
                    'tob' => $request->tob,
                    'pob' => $request->pob,
                    'gender' => $request->gender,

                    // --- THE NEW ADDITION ---
                    // Reset the cached chart data to NULL whenever birth details change.
                    // This forces the Dashboard to fetch a fresh, accurate chart next time.
                    'chart_data' => null,
                ]
            );
        }

        return back()->with('success', 'Profile and Kundali details updated successfully.');
    }
}
