<?php

namespace App\Filament\Resources;

use App\Filament\Resources\TransactionResource\Pages;
use App\Models\Transaction;
use Filament\Forms;
use Filament\Forms\Form;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Table;

class TransactionResource extends Resource
{
    protected static ?string $model = Transaction::class;
    protected static ?string $navigationIcon = 'heroicon-o-banknotes';
    protected static ?string $navigationGroup = 'Finance';

    public static function form(Form $form): Form
    {
        return $form
            ->schema([
                Forms\Components\Section::make('Payment Info')
                    ->schema([
                        Forms\Components\Select::make('booking_id')
                            ->relationship('booking', 'id') // Links to booking reference
                            ->required(),

                        Forms\Components\TextInput::make('transaction_id')
                            ->label('Gateway Reference ID')
                            ->required()
                            ->unique(ignoreRecord: true),

                        Forms\Components\TextInput::make('amount')
                            ->numeric()
                            ->prefix('NPR')
                            ->required(),

                        Forms\Components\Select::make('payment_method')
                            ->options([
                                'esewa' => 'eSewa',
                                'khalti' => 'Khalti',
                                'bank_transfer' => 'Bank Transfer',
                            ])->required(),

                        Forms\Components\Select::make('payment_status')
                            ->options([
                                'pending' => 'Pending',
                                'success' => 'Success',
                                'failed' => 'Failed',
                                'refunded' => 'Refunded',
                            ])->required(),
                    ])->columns(2),

                Forms\Components\KeyValue::make('payload')
                    ->label('Gateway API Response')
                    ->helperText('Technical data from the payment provider'),
            ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                Tables\Columns\TextColumn::make('transaction_id')->label('Ref ID')->searchable(),
                Tables\Columns\TextColumn::make('booking.user.name')->label('Customer'),
                Tables\Columns\TextColumn::make('amount')->money('NPR'),
                Tables\Columns\BadgeColumn::make('payment_status')
                    ->colors([
                        'warning' => 'pending',
                        'success' => 'success',
                        'danger' => 'failed',
                        'info' => 'refunded',
                    ]),
                Tables\Columns\TextColumn::make('created_at')->dateTime(),
            ]);
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListTransactions::route('/'),
            'create' => Pages\CreateTransaction::route('/create'),
            'edit' => Pages\EditTransaction::route('/{record}/edit'),
        ];
    }
}
